/*
* mhal_pinmux.c- Sigmastar
*
* Copyright (C) 2018 Sigmastar Technology Corp.
*
* Author: karl.xiao <karl.xiao@sigmastar.com.tw>
*
* This software is licensed under the terms of the GNU General Public
* License version 2, as published by the Free Software Foundation, and
* may be copied, distributed, and modified under those terms.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
*/
#include "ms_platform.h"
#include "mdrv_types.h"
#include "mhal_gpio.h"
#include "padmux.h"
#include "gpio.h"

//==============================================================================
//
//                              MACRO DEFINE
//
//==============================================================================

#define BASE_RIU_PA                         0xFD000000
#define PMSLEEP_BANK                        0x000E00
#define SAR_BANK                            0x001400
#define ALBANY1_BANK                        0x003200
#define ALBANY2_BANK                        0x003300
#define CHIPTOP_BANK                        0x101E00
#define UTMI0_BANK                          0x142100
#define UTMI1_BANK                          0x142900

#define _GPIO_W_WORD(addr,val)              {(*(volatile u16*)(addr)) = (u16)(val);}
#define _GPIO_W_WORD_MASK(addr,val,mask)    {(*(volatile u16*)(addr)) = ((*(volatile u16*)(addr)) & ~(mask)) | ((u16)(val) & (mask));}
#define _GPIO_R_BYTE(addr)                  (*(volatile u8*)(addr))
#define _GPIO_R_WORD_MASK(addr,mask)        ((*(volatile u16*)(addr)) & (mask))

#define GET_BASE_ADDR_BY_BANK(x, y)         ((x) + ((y) << 1))
#define _RIUA_8BIT(bank , offset)           GET_BASE_ADDR_BY_BANK(BASE_RIU_PA, bank) + (((offset) & ~1)<<1) + ((offset) & 1)
#define _RIUA_16BIT(bank , offset)          GET_BASE_ADDR_BY_BANK(BASE_RIU_PA, bank) + ((offset)<<2)

/* Non PM Pad : CHIPTOP_BANK */
#define REG_FUART_MODE          0x03
    #define REG_FUART_MODE_MASK     BIT2|BIT1|BIT0
#define REG_UART0_MODE          0x03
    #define REG_UART0_MODE_MASK     BIT6|BIT5|BIT4
#define REG_UART1_MODE          0x03
    #define REG_UART1_MODE_MASK     BIT9|BIT8
#define REG_PWM0_MODE           0x04
    #define REG_PWM0_MODE_MASK      BIT2|BIT1|BIT0
#define REG_PWM1_MODE           0x04
    #define REG_PWM1_MODE_MASK      BIT5|BIT4|BIT3
#define REG_PWM2_MODE           0x04
    #define REG_PWM2_MODE_MASK      BIT10|BIT9|BIT8
#define REG_PWM3_MODE           0x04
    #define REG_PWM3_MODE_MASK      BIT14|BIT13|BIT12
#define REG_PWM4_MODE           0x05
    #define REG_PWM4_MODE_MASK      BIT1|BIT0
#define REG_PWM5_MODE           0x05
    #define REG_PWM5_MODE_MASK      BIT4|BIT3
#define REG_PWM6_MODE           0x05
    #define REG_PWM6_MODE_MASK      BIT9|BIT8
#define REG_PWM7_MODE           0x05
    #define REG_PWM7_MODE_MASK      BIT13|BIT12
#define REG_SR_MODE             0x06
    #define REG_SR_MODE_MASK        BIT2|BIT1|BIT0
#define REG_NAND_MODE           0x08
    #define REG_NAND_MODE_MASK      BIT0
#define REG_SD_MODE             0x08
    #define REG_SD_MODE_MASK        BIT3|BIT2
#define REG_SDIO_MODE           0x08
    #define REG_SDIO_MODE_MASK      BIT10|BIT9|BIT8
#define REG_I2C0_MODE           0x09
    #define REG_I2C0_MODE_MASK      BIT1|BIT0
#define REG_I2C1_MODE           0x09
    #define REG_I2C1_MODE_MASK      BIT6|BIT5|BIT4
#define REG_SPI0_MODE           0x0c
    #define REG_SPI0_MODE_MASK      BIT1|BIT0
#define REG_SPI1_MODE           0x0c
    #define REG_SPI1_MODE_MASK      BIT6|BIT5|BIT4
#define REG_EJ_MODE             0x0f
    #define REG_EJ_MODE_MASK        BIT1|BIT0
#define REG_ETH_MODE            0x0f
    #define REG_ETH_MODE_MASK       BIT2
#define REG_TTL_MODE            0x0f
    #define REG_TTL_MODE_MASK       BIT7|BIT6
#define REG_DMIC_MODE           0x0f
    #define REG_DMIC_MODE_MASK      BIT10|BIT9|BIT8
#define REG_I2S_MODE            0x0f
    #define REG_I2S_MODE_MASK       BIT13|BIT12
#define REG_I2S_RX_MODE         0x10
    #define REG_I2S_RX_MODE_MASK    BIT0
#define REG_I2S_TX_MODE         0x10
    #define REG_I2S_TX_MODE_MASK    BIT4
#define REG_I2S_MCK_MODE        0x10
    #define REG_I2S_MCK_MODE_MASK   BIT8
#define REG_TESTIN_MODE         0x12
    #define REG_TESTIN_MODE_MASK    BIT1|BIT0
#define REG_TESTOUT_MODE        0x12
    #define REG_TESTOUT_MODE_MASK   BIT5|BIT4
#define REG_EMMC_MODE           0x13
    #define REG_EMMC_MODE_MASK      BIT0
#define REG_EMMC_RSTN_EN        0x13
    #define REG_EMMC_RSTN_EN_MASK   BIT1
#define REG_SR0_BT656_MODE      0x15
    #define REG_SR0_BT656_MODE_MASK BIT6|BIT5|BIT4
#define REG_SR0_MIPI_MODE       0x15
    #define REG_SR0_MIPI_MODE_MASK  BIT9|BIT8|BIT7
#define REG_SR0_PAR_MODE        0x15
    #define REG_SR0_PAR_MODE_MASK   BIT12|BIT11|BIT10
#define REG_I2C2_MODE           0x16
    #define REG_I2C2_MODE_MASK      BIT9|BIT8
#define REG_I2C3_MODE           0x16
    #define REG_I2C3_MODE_MASK      BIT12|BIT11|BIT10
#define REG_SR1_BT656_MODE      0x18
    #define REG_SR1_BT656_MODE_MASK BIT4
#define REG_SR1_MIPI_MODE       0x18
    #define REG_SR1_MIPI_MODE_MASK  BIT8|BIT7
#define REG_SR1_PAR_MODE        0x18
    #define REG_SR1_PAR_MODE_MASK   BIT10

/* PM Sleep : PMSLEEP_BANK */
#define REG_GPIO_PM_LOCK        0x12
    #define REG_GPIO_PM_LOCK_MASK   0xFFFF
#define REG_PM_GPIO_PM4_INV     0x1c
    #define REG_PM_GPIO_PM4_INV_MASK BIT1
#define REG_PM_LINK_WKINT2GPIO4 0x1c
    #define REG_PM_LINK_WKINT2GPIO4_MASK BIT3
#define REG_PM_IR_IS_GPIO       0x1c
    #define REG_PM_IR_IS_GPIO_MASK  BIT4
#define REG_PM_PWM0_MODE        0x28
    #define REG_PM_PWM0_MODE_MASK   BIT1|BIT0
#define REG_PM_PWM1_MODE        0x28
    #define REG_PM_PWM1_MODE_MASK   BIT3|BIT2
#define REG_PM_LED_MODE         0x28
    #define REG_PM_LED_MODE_MASK    BIT5|BIT4
#define REG_PM_PWM2_MODE        0x28
    #define REG_PM_PWM2_MODE_MASK   BIT7|BIT6
#define REG_PM_PWM3_MODE        0x28
    #define REG_PM_PWM3_MODE_MASK   BIT9|BIT8
#define REG_PM_VID_MODE         0x28
    #define REG_PM_VID_MODE_MASK    BIT13|BIT12
#define REG_PM_SD_CDZ_MODE      0x28
    #define REG_PM_SD_CDZ_MODE_MASK BIT14
#define REG_PM_SPI_IS_GPIO      0x35
    #define REG_PM_SPI_IS_GPIO_MASK BIT7|BIT6|BIT5|BIT4|BIT3|BIT2|BIT1|BIT0
#define REG_PM_UART_IS_GPIO     0x35
    #define REG_PM_UART_IS_GPIO_MASK BIT9|BIT8

/* SAR : SAR_BANK, R/W 8-bits */
#define REG_SAR_AISEL           0x11*2
    #define REG_SAR_CH0_AISEL       BIT0
    #define REG_SAR_CH1_AISEL       BIT1
    #define REG_SAR_CH2_AISEL       BIT2
    #define REG_SAR_CH3_AISEL       BIT3

/* EMAC : ALBANY1_BANK */
#define REG_ATOP_RX_INOFF       0x69
    #define REG_ATOP_RX_INOFF_MASK  BIT15|BIT14

/* EMAC : ALBANY2_BANK */
#define REG_ETH_GPIO_EN         0x71
    #define REG_ETH_GPIO_EN_MASK    BIT3|BIT2|BIT1|BIT0

/* UTMI0 : UTMI0_BANK */
#define REG_UTMI0_FL_XVR_PDN    0x0
    #define REG_UTMI0_FL_XVR_PDN_MASK BIT12
#define REG_UTMI0_REG_PDN       0x0
    #define REG_UTMI0_REG_PDN_MASK  BIT15
#define REG_UTMI0_CLK_EXTRA0_EN 0x4
    #define REG_UTMI0_CLK_EXTRA0_EN_MASK BIT7
#define REG_UTMI0_GPIO_EN       0x1f
    #define REG_UTMI0_GPIO_EN_MASK  BIT14

/* UTMI1 : UTMI1_BANK */
#define REG_UTMI1_FL_XVR_PDN    0x0
    #define REG_UTMI1_FL_XVR_PDN_MASK  BIT12
#define REG_UTMI1_REG_PDN       0x0
    #define REG_UTMI1_REG_PDN_MASK  BIT15
#define REG_UTMI1_CLK_EXTRA0_EN 0x4
    #define REG_UTMI1_CLK_EXTRA0_EN_MASK BIT7
#define REG_UTMI1_GPIO_EN       0x1f
    #define REG_UTMI1_GPIO_EN_MASK  BIT14

//==============================================================================
//
//                              STRUCTURE
//
//==============================================================================

typedef struct stPadMux
{
    U16 padID;
    U32 base;
    U16 offset;
    U16 mask;
    U16 val;
    U16 mode;
} ST_PadMuxInfo;

typedef struct stPadMode
{
    U8  u8PadName[16];
    U32 u32ModeRIU;
    U32 u32ModeMask;
} ST_PadModeInfo;

//==============================================================================
//
//                              VARIABLES
//
//==============================================================================

static const ST_PadMuxInfo m_stNonPMPadMuxTbl[] =
{
    {PAD_GPIO0, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT6,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO0, CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,    BIT1,           PINMUX_FOR_FUART_MODE},
    {PAD_GPIO0, CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,      BIT12,          PINMUX_FOR_I2S_MODE},

    {PAD_GPIO1, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT6,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO1, CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,    BIT1,           PINMUX_FOR_FUART_MODE},
    {PAD_GPIO1, CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,      BIT12,          PINMUX_FOR_I2S_MODE},

    {PAD_GPIO2, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT6,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO2, CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,    BIT1,           PINMUX_FOR_FUART_MODE},
    {PAD_GPIO2, CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,      BIT12,          PINMUX_FOR_I2S_MODE},

    {PAD_GPIO3, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT6,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO3, CHIPTOP_BANK, REG_FUART_MODE,           REG_FUART_MODE_MASK,    BIT1,           PINMUX_FOR_FUART_MODE},
    {PAD_GPIO3, CHIPTOP_BANK, REG_I2S_MODE,             REG_I2S_MODE_MASK,      BIT12,          PINMUX_FOR_I2S_MODE},

    {PAD_GPIO4, CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,    BIT5|BIT4,      PINMUX_FOR_UART0_MODE},
    {PAD_GPIO4, CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,     BIT8,           PINMUX_FOR_DMIC_MODE},

    {PAD_GPIO5, CHIPTOP_BANK, REG_UART0_MODE,           REG_UART0_MODE_MASK,    BIT5|BIT4,      PINMUX_FOR_UART0_MODE},
    {PAD_GPIO5, CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,     BIT8,           PINMUX_FOR_DMIC_MODE},

    {PAD_GPIO6, CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,    BIT9|BIT8,      PINMUX_FOR_UART1_MODE},
    {PAD_GPIO6, CHIPTOP_BANK, REG_DMIC_MODE,            REG_DMIC_MODE_MASK,     BIT8,           PINMUX_FOR_DMIC_MODE},

    {PAD_GPIO7, CHIPTOP_BANK, REG_UART1_MODE,           REG_UART1_MODE_MASK,    BIT9|BIT8,      PINMUX_FOR_UART1_MODE},
    {PAD_GPIO7, CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO8, CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,     BIT1,           PINMUX_FOR_SPI0_MODE},
    {PAD_GPIO8, CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO9, CHIPTOP_BANK, REG_SPI0_MODE,            REG_SPI0_MODE_MASK,     BIT1,           PINMUX_FOR_SPI0_MODE},
    {PAD_GPIO9, CHIPTOP_BANK, REG_ETH_MODE,             REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO10, CHIPTOP_BANK, REG_SPI0_MODE,           REG_SPI0_MODE_MASK,     BIT1,           PINMUX_FOR_SPI0_MODE},
    {PAD_GPIO10, CHIPTOP_BANK, REG_ETH_MODE,            REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO11, CHIPTOP_BANK, REG_SPI0_MODE,           REG_SPI0_MODE_MASK,     BIT1,           PINMUX_FOR_SPI0_MODE},
    {PAD_GPIO11, CHIPTOP_BANK, REG_ETH_MODE,            REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO12, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT5,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO12, CHIPTOP_BANK, REG_PWM0_MODE,           REG_PWM0_MODE_MASK,     BIT1,           PINMUX_FOR_PWM0_MODE},
    {PAD_GPIO12, CHIPTOP_BANK, REG_ETH_MODE,            REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO13, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT5,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO13, CHIPTOP_BANK, REG_PWM1_MODE,           REG_PWM1_MODE_MASK,     BIT4,           PINMUX_FOR_PWM1_MODE},
    {PAD_GPIO13, CHIPTOP_BANK, REG_ETH_MODE,            REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO14, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT5,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO14, CHIPTOP_BANK, REG_PWM2_MODE,           REG_PWM2_MODE_MASK,     BIT8,           PINMUX_FOR_PWM2_MODE},
    {PAD_GPIO14, CHIPTOP_BANK, REG_ETH_MODE,            REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_GPIO15, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT5,           PINMUX_FOR_SPI1_MODE},
    {PAD_GPIO15, CHIPTOP_BANK, REG_FUART_MODE,          REG_FUART_MODE_MASK,    BIT1|BIT0,      PINMUX_FOR_FUART_MODE},
    {PAD_GPIO15, CHIPTOP_BANK, REG_PWM3_MODE,           REG_PWM3_MODE_MASK,     BIT12,          PINMUX_FOR_PWM3_MODE},
    {PAD_GPIO15, CHIPTOP_BANK, REG_ETH_MODE,            REG_ETH_MODE_MASK,      BIT2,           PINMUX_FOR_ETH_MODE},

    {PAD_FUART_RX, CHIPTOP_BANK, REG_EJ_MODE,           REG_EJ_MODE_MASK,       BIT0,           PINMUX_FOR_EJ_MODE},
    {PAD_FUART_RX, CHIPTOP_BANK, REG_TESTIN_MODE,       REG_TESTIN_MODE_MASK,   BIT1|BIT0,      PINMUX_FOR_TESTIN_MODE},
    {PAD_FUART_RX, CHIPTOP_BANK, REG_TESTOUT_MODE,      REG_TESTOUT_MODE_MASK,  BIT4,           PINMUX_FOR_TESTOUT_MODE},
    {PAD_FUART_RX, CHIPTOP_BANK, REG_SPI0_MODE,         REG_SPI0_MODE_MASK,     BIT1|BIT0,      PINMUX_FOR_SPI0_MODE},
    {PAD_FUART_RX, CHIPTOP_BANK, REG_FUART_MODE,        REG_FUART_MODE_MASK,    BIT1|BIT0,      PINMUX_FOR_FUART_MODE},
    {PAD_FUART_RX, CHIPTOP_BANK, REG_UART0_MODE,        REG_UART0_MODE_MASK,    BIT5,           PINMUX_FOR_UART0_MODE},
    {PAD_FUART_RX, CHIPTOP_BANK, REG_PWM0_MODE,         REG_PWM0_MODE_MASK,     BIT1|BIT0,      PINMUX_FOR_PWM0_MODE},

    {PAD_FUART_TX, CHIPTOP_BANK, REG_EJ_MODE,           REG_EJ_MODE_MASK,       BIT0,           PINMUX_FOR_EJ_MODE},
    {PAD_FUART_TX, CHIPTOP_BANK, REG_TESTIN_MODE,       REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_FUART_TX, CHIPTOP_BANK, REG_TESTOUT_MODE,      REG_TESTOUT_MODE_MASK,  BIT4,           PINMUX_FOR_TESTOUT_MODE},
    {PAD_FUART_TX, CHIPTOP_BANK, REG_SPI0_MODE,         REG_SPI0_MODE_MASK,     BIT1|BIT0,      PINMUX_FOR_SPI0_MODE},
    {PAD_FUART_TX, CHIPTOP_BANK, REG_FUART_MODE,        REG_FUART_MODE_MASK,    BIT1|BIT0,      PINMUX_FOR_FUART_MODE},
    {PAD_FUART_TX, CHIPTOP_BANK, REG_UART0_MODE,        REG_UART0_MODE_MASK,    BIT5,           PINMUX_FOR_UART0_MODE},
    {PAD_FUART_TX, CHIPTOP_BANK, REG_PWM1_MODE,         REG_PWM1_MODE_MASK,     BIT4|BIT3,      PINMUX_FOR_PWM1_MODE},

    {PAD_FUART_CTS, CHIPTOP_BANK, REG_EJ_MODE,          REG_EJ_MODE_MASK,       BIT0,           PINMUX_FOR_EJ_MODE},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_TESTIN_MODE,      REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_TESTOUT_MODE,     REG_TESTOUT_MODE_MASK,  BIT4,           PINMUX_FOR_TESTOUT_MODE},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_SPI0_MODE,        REG_SPI0_MODE_MASK,     BIT1|BIT0,      PINMUX_FOR_SPI0_MODE},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_FUART_MODE,       REG_FUART_MODE_MASK,    BIT1|BIT0,      PINMUX_FOR_FUART_MODE},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_UART1_MODE,       REG_UART1_MODE_MASK,    BIT9,           PINMUX_FOR_UART1_MODE},
    {PAD_FUART_CTS, CHIPTOP_BANK, REG_PWM2_MODE,        REG_PWM2_MODE_MASK,     BIT9,           PINMUX_FOR_PWM2_MODE},

    {PAD_FUART_RTS, CHIPTOP_BANK, REG_EJ_MODE,          REG_EJ_MODE_MASK,       BIT0,           PINMUX_FOR_EJ_MODE},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_TESTIN_MODE,      REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_TESTOUT_MODE,     REG_TESTOUT_MODE_MASK,  BIT4,           PINMUX_FOR_TESTOUT_MODE},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_SPI0_MODE,        REG_SPI0_MODE_MASK,     BIT1|BIT0,      PINMUX_FOR_SPI0_MODE},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_FUART_MODE,       REG_FUART_MODE_MASK,    BIT0,           PINMUX_FOR_FUART_MODE},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_UART1_MODE,       REG_UART1_MODE_MASK,    BIT9,           PINMUX_FOR_UART1_MODE},
    {PAD_FUART_RTS, CHIPTOP_BANK, REG_PWM3_MODE,        REG_PWM3_MODE_MASK,     BIT13,          PINMUX_FOR_PWM3_MODE},

    {PAD_I2C0_SCL, CHIPTOP_BANK, REG_I2C0_MODE,         REG_I2C0_MODE_MASK,     BIT0,           PINMUX_FOR_I2C0_MODE},
    {PAD_I2C0_SCL, CHIPTOP_BANK, REG_I2C3_MODE,         REG_I2C3_MODE_MASK,     BIT10,          PINMUX_FOR_I2C3_MODE},

    {PAD_I2C0_SDA, CHIPTOP_BANK, REG_I2C0_MODE,         REG_I2C0_MODE_MASK,     BIT0,           PINMUX_FOR_I2C0_MODE},
    {PAD_I2C0_SDA, CHIPTOP_BANK, REG_I2C3_MODE,         REG_I2C3_MODE_MASK,     BIT10,          PINMUX_FOR_I2C3_MODE},

    {PAD_I2C1_SCL, CHIPTOP_BANK, REG_TESTIN_MODE,       REG_TESTIN_MODE_MASK,   BIT1,           PINMUX_FOR_TESTIN_MODE},
    {PAD_I2C1_SCL, CHIPTOP_BANK, REG_TESTOUT_MODE,      REG_TESTOUT_MODE_MASK,  BIT5,           PINMUX_FOR_TESTOUT_MODE},
    {PAD_I2C1_SCL, CHIPTOP_BANK, REG_I2C1_MODE,         REG_I2C1_MODE_MASK,     BIT4,           PINMUX_FOR_I2C1_MODE},

    {PAD_I2C1_SDA, CHIPTOP_BANK, REG_TESTIN_MODE,       REG_TESTIN_MODE_MASK,   BIT1,           PINMUX_FOR_TESTIN_MODE},
    {PAD_I2C1_SDA, CHIPTOP_BANK, REG_TESTOUT_MODE,      REG_TESTOUT_MODE_MASK,  BIT5,           PINMUX_FOR_TESTOUT_MODE},
    {PAD_I2C1_SDA, CHIPTOP_BANK, REG_I2C1_MODE,         REG_I2C1_MODE_MASK,     BIT4,           PINMUX_FOR_I2C1_MODE},

    {PAD_SNR0_D0, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11,    PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D0, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT5|BIT4,      PINMUX_FOR_SR0_BT656_MODE},
    {PAD_SNR0_D0, CHIPTOP_BANK, REG_I2C0_MODE,          REG_I2C0_MODE_MASK,     BIT1|BIT0,      PINMUX_FOR_I2C0_MODE},

    {PAD_SNR0_D1, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D1, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D1, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT5|BIT4,      PINMUX_FOR_SR0_BT656_MODE},
    {PAD_SNR0_D1, CHIPTOP_BANK, REG_I2C0_MODE,          REG_I2C0_MODE_MASK,     BIT1|BIT0,      PINMUX_FOR_I2C0_MODE},

    {PAD_SNR0_D2, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D2, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D2, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D2, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D2, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT5|BIT4,      PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D3, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D3, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D3, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D3, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D3, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D4, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D4, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D4, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D4, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D4, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D5, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D5, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D5, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D5, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D5, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D6, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D6, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D6, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D6, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D6, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D7, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D7, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D7, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D7, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D7, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D8, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D8, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D8, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT8|BIT7, PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D8, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D8, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D9, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT0,           PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D9, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT5|BIT4,      PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D9, CHIPTOP_BANK, REG_SR0_MIPI_MODE,      REG_SR0_MIPI_MODE_MASK, BIT9|BIT7,      PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D9, CHIPTOP_BANK, REG_SR0_PAR_MODE,       REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D9, CHIPTOP_BANK, REG_SR0_BT656_MODE,     REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D10, CHIPTOP_BANK, REG_TESTIN_MODE,       REG_TESTIN_MODE_MASK,   BIT0,          PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D10, CHIPTOP_BANK, REG_TESTOUT_MODE,      REG_TESTOUT_MODE_MASK,  BIT5|BIT4,     PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D10, CHIPTOP_BANK, REG_SR0_MIPI_MODE,     REG_SR0_MIPI_MODE_MASK, BIT9|BIT7,     PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_D10, CHIPTOP_BANK, REG_SR0_PAR_MODE,      REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D10, CHIPTOP_BANK, REG_SR0_BT656_MODE,    REG_SR0_BT656_MODE_MASK,BIT6|BIT4,     PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_D11, CHIPTOP_BANK, REG_TESTIN_MODE,       REG_TESTIN_MODE_MASK,   BIT0,          PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_D11, CHIPTOP_BANK, REG_TESTOUT_MODE,      REG_TESTOUT_MODE_MASK,  BIT5|BIT4,     PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_D11, CHIPTOP_BANK, REG_SR0_PAR_MODE,      REG_SR0_PAR_MODE_MASK,  BIT12|BIT11,   PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_D11, CHIPTOP_BANK, REG_SR0_BT656_MODE,    REG_SR0_BT656_MODE_MASK,BIT4,          PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_GPIO0, CHIPTOP_BANK, REG_TESTIN_MODE,     REG_TESTIN_MODE_MASK,   BIT0,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_GPIO0, CHIPTOP_BANK, REG_TESTOUT_MODE,    REG_TESTOUT_MODE_MASK,  BIT5|BIT4,  PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_GPIO0, CHIPTOP_BANK, REG_SR0_MIPI_MODE,   REG_SR0_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_GPIO0, CHIPTOP_BANK, REG_SR0_PAR_MODE,    REG_SR0_PAR_MODE_MASK,  BIT12|BIT11,PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_GPIO0, CHIPTOP_BANK, REG_SR0_BT656_MODE,  REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_GPIO1, CHIPTOP_BANK, REG_TESTIN_MODE,     REG_TESTIN_MODE_MASK,   BIT0,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_GPIO1, CHIPTOP_BANK, REG_TESTOUT_MODE,    REG_TESTOUT_MODE_MASK,  BIT5|BIT4,  PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_GPIO1, CHIPTOP_BANK, REG_SR0_MIPI_MODE,   REG_SR0_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_GPIO1, CHIPTOP_BANK, REG_SR0_PAR_MODE,    REG_SR0_PAR_MODE_MASK,  BIT12|BIT11,PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_GPIO1, CHIPTOP_BANK, REG_SR0_BT656_MODE,  REG_SR0_BT656_MODE_MASK,BIT5|BIT4,  PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_GPIO2, CHIPTOP_BANK, REG_TESTIN_MODE,     REG_TESTIN_MODE_MASK,   BIT0,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_GPIO2, CHIPTOP_BANK, REG_TESTOUT_MODE,    REG_TESTOUT_MODE_MASK,  BIT5|BIT4,  PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_GPIO2, CHIPTOP_BANK, REG_SR0_MIPI_MODE,   REG_SR0_MIPI_MODE_MASK, BIT9|BIT7,  PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_GPIO2, CHIPTOP_BANK, REG_SR0_PAR_MODE,    REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_GPIO2, CHIPTOP_BANK, REG_SR0_BT656_MODE,  REG_SR0_BT656_MODE_MASK,BIT6,       PINMUX_FOR_SR0_BT656_MODE},
    {PAD_SNR0_GPIO2, CHIPTOP_BANK, REG_I2C1_MODE,       REG_I2C1_MODE_MASK,     BIT5|BIT4,  PINMUX_FOR_I2C1_MODE},

    {PAD_SNR0_GPIO3, CHIPTOP_BANK, REG_TESTIN_MODE,     REG_TESTIN_MODE_MASK,   BIT0,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_GPIO3, CHIPTOP_BANK, REG_TESTOUT_MODE,    REG_TESTOUT_MODE_MASK,  BIT5|BIT4,  PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_GPIO3, CHIPTOP_BANK, REG_SR0_MIPI_MODE,   REG_SR0_MIPI_MODE_MASK, BIT9|BIT7,  PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_GPIO3, CHIPTOP_BANK, REG_SR0_PAR_MODE,    REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_GPIO3, CHIPTOP_BANK, REG_SR0_BT656_MODE,  REG_SR0_BT656_MODE_MASK,BIT6|BIT5|BIT4, PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_GPIO4, CHIPTOP_BANK, REG_TESTIN_MODE,     REG_TESTIN_MODE_MASK,   BIT0,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_GPIO4, CHIPTOP_BANK, REG_TESTOUT_MODE,    REG_TESTOUT_MODE_MASK,  BIT5|BIT4,  PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_GPIO4, CHIPTOP_BANK, REG_SR0_MIPI_MODE,   REG_SR0_MIPI_MODE_MASK, BIT9|BIT7,  PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_GPIO4, CHIPTOP_BANK, REG_SR0_PAR_MODE,    REG_SR0_PAR_MODE_MASK,  BIT12|BIT11|BIT10, PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_GPIO4, CHIPTOP_BANK, REG_SR0_BT656_MODE,  REG_SR0_BT656_MODE_MASK,BIT6,       PINMUX_FOR_SR0_BT656_MODE},
    {PAD_SNR0_GPIO4, CHIPTOP_BANK, REG_I2C1_MODE,       REG_I2C1_MODE_MASK,     BIT5|BIT4,  PINMUX_FOR_I2C1_MODE},
    {PAD_SNR0_GPIO4, CHIPTOP_BANK, REG_PWM2_MODE,       REG_PWM2_MODE_MASK,     BIT10|BIT8, PINMUX_FOR_PWM2_MODE},

    {PAD_SNR0_GPIO5, CHIPTOP_BANK, REG_TESTIN_MODE,     REG_TESTIN_MODE_MASK,   BIT0,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SNR0_GPIO5, CHIPTOP_BANK, REG_TESTOUT_MODE,    REG_TESTOUT_MODE_MASK,  BIT5|BIT4,  PINMUX_FOR_TESTOUT_MODE},
    {PAD_SNR0_GPIO5, CHIPTOP_BANK, REG_SR0_MIPI_MODE,   REG_SR0_MIPI_MODE_MASK, BIT9|BIT7,  PINMUX_FOR_SR0_MIPI_MODE},
    {PAD_SNR0_GPIO5, CHIPTOP_BANK, REG_SR0_PAR_MODE,    REG_SR0_PAR_MODE_MASK,  BIT12|BIT11,PINMUX_FOR_SR0_PAR_MODE},
    {PAD_SNR0_GPIO5, CHIPTOP_BANK, REG_SR0_BT656_MODE,  REG_SR0_BT656_MODE_MASK,BIT5|BIT4,  PINMUX_FOR_SR0_BT656_MODE},

    {PAD_SNR0_GPIO6, CHIPTOP_BANK, REG_PWM2_MODE,       REG_PWM2_MODE_MASK,     BIT10,      PINMUX_FOR_PWM2_MODE},

    {PAD_SNR1_DA0P, CHIPTOP_BANK, REG_SR1_MIPI_MODE,    REG_SR1_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_DA0P, CHIPTOP_BANK, REG_SR1_PAR_MODE,     REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_DA0P, CHIPTOP_BANK, REG_SR1_BT656_MODE,   REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_DA0P, CHIPTOP_BANK, REG_SPI1_MODE,        REG_SPI1_MODE_MASK,     BIT6|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SNR1_DA0P, CHIPTOP_BANK, REG_UART0_MODE,       REG_UART0_MODE_MASK,    BIT6,       PINMUX_FOR_UART0_MODE},

    {PAD_SNR1_DA0N, CHIPTOP_BANK, REG_SR1_MIPI_MODE,    REG_SR1_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_DA0N, CHIPTOP_BANK, REG_SR1_PAR_MODE,     REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_DA0N, CHIPTOP_BANK, REG_SR1_BT656_MODE,   REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_DA0N, CHIPTOP_BANK, REG_SPI1_MODE,        REG_SPI1_MODE_MASK,     BIT6|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SNR1_DA0N, CHIPTOP_BANK, REG_UART0_MODE,       REG_UART0_MODE_MASK,    BIT6,       PINMUX_FOR_UART0_MODE},

    {PAD_SNR1_CKP, CHIPTOP_BANK, REG_EJ_MODE,           REG_EJ_MODE_MASK,       BIT1|BIT0,  PINMUX_FOR_EJ_MODE},
    {PAD_SNR1_CKP, CHIPTOP_BANK, REG_SR1_MIPI_MODE,     REG_SR1_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_CKP, CHIPTOP_BANK, REG_SR1_PAR_MODE,      REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_CKP, CHIPTOP_BANK, REG_SR1_BT656_MODE,    REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_CKP, CHIPTOP_BANK, REG_SPI1_MODE,         REG_SPI1_MODE_MASK,     BIT6|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SNR1_CKP, CHIPTOP_BANK, REG_FUART_MODE,        REG_FUART_MODE_MASK,    BIT2|BIT0,  PINMUX_FOR_FUART_MODE},

    {PAD_SNR1_CKN, CHIPTOP_BANK, REG_EJ_MODE,           REG_EJ_MODE_MASK,       BIT1|BIT0,  PINMUX_FOR_EJ_MODE},
    {PAD_SNR1_CKN, CHIPTOP_BANK, REG_SR1_MIPI_MODE,     REG_SR1_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_CKN, CHIPTOP_BANK, REG_SR1_PAR_MODE,      REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_CKN, CHIPTOP_BANK, REG_SR1_BT656_MODE,    REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_CKN, CHIPTOP_BANK, REG_SPI1_MODE,         REG_SPI1_MODE_MASK,     BIT6|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SNR1_CKN, CHIPTOP_BANK, REG_FUART_MODE,        REG_FUART_MODE_MASK,    BIT2|BIT0,  PINMUX_FOR_FUART_MODE},

    {PAD_SNR1_DA1P, CHIPTOP_BANK, REG_EJ_MODE,          REG_EJ_MODE_MASK,       BIT1|BIT0,  PINMUX_FOR_EJ_MODE},
    {PAD_SNR1_DA1P, CHIPTOP_BANK, REG_SR1_MIPI_MODE,    REG_SR1_MIPI_MODE_MASK, BIT7,       PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_DA1P, CHIPTOP_BANK, REG_SR1_PAR_MODE,     REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_DA1P, CHIPTOP_BANK, REG_SR1_BT656_MODE,   REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_DA1P, CHIPTOP_BANK, REG_FUART_MODE,       REG_FUART_MODE_MASK,    BIT2|BIT0,  PINMUX_FOR_FUART_MODE},

    {PAD_SNR1_DA1N, CHIPTOP_BANK, REG_EJ_MODE,          REG_EJ_MODE_MASK,       BIT1|BIT0,  PINMUX_FOR_EJ_MODE},
    {PAD_SNR1_DA1N, CHIPTOP_BANK, REG_SR1_MIPI_MODE,    REG_SR1_MIPI_MODE_MASK, BIT7,       PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_DA1N, CHIPTOP_BANK, REG_SR1_PAR_MODE,     REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_DA1N, CHIPTOP_BANK, REG_SR1_BT656_MODE,   REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_DA1N, CHIPTOP_BANK, REG_FUART_MODE,       REG_FUART_MODE_MASK,    BIT2|BIT0,  PINMUX_FOR_FUART_MODE},

    {PAD_SNR1_GPIO0, CHIPTOP_BANK, REG_SR1_PAR_MODE,    REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_GPIO0, CHIPTOP_BANK, REG_SR1_BT656_MODE,  REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_GPIO0, CHIPTOP_BANK, REG_SDIO_MODE,       REG_SDIO_MODE_MASK,     BIT10,      PINMUX_FOR_SDIO_MODE},

    {PAD_SNR1_GPIO1, CHIPTOP_BANK, REG_SR1_MIPI_MODE,   REG_SR1_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_GPIO1, CHIPTOP_BANK, REG_SR1_PAR_MODE,    REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_GPIO1, CHIPTOP_BANK, REG_SR1_BT656_MODE,  REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_GPIO1, CHIPTOP_BANK, REG_SDIO_MODE,       REG_SDIO_MODE_MASK,     BIT10,      PINMUX_FOR_SDIO_MODE},

    {PAD_SNR1_GPIO2, CHIPTOP_BANK, REG_SR1_PAR_MODE,    REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_GPIO2, CHIPTOP_BANK, REG_SR1_BT656_MODE,  REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_GPIO2, CHIPTOP_BANK, REG_SDIO_MODE,       REG_SDIO_MODE_MASK,     BIT10,      PINMUX_FOR_SDIO_MODE},

    {PAD_SNR1_GPIO3, CHIPTOP_BANK, REG_SR1_MIPI_MODE,   REG_SR1_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_GPIO3, CHIPTOP_BANK, REG_SR1_PAR_MODE,    REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_GPIO3, CHIPTOP_BANK, REG_SR1_BT656_MODE,  REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_GPIO3, CHIPTOP_BANK, REG_SDIO_MODE,       REG_SDIO_MODE_MASK,     BIT10,      PINMUX_FOR_SDIO_MODE},

    {PAD_SNR1_GPIO4, CHIPTOP_BANK, REG_SR1_MIPI_MODE,   REG_SR1_MIPI_MODE_MASK, BIT8|BIT7,  PINMUX_FOR_SR1_MIPI_MODE},
    {PAD_SNR1_GPIO4, CHIPTOP_BANK, REG_SR1_PAR_MODE,    REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_GPIO4, CHIPTOP_BANK, REG_SR1_BT656_MODE,  REG_SR1_BT656_MODE_MASK,BIT4,       PINMUX_FOR_SR1_BT656_MODE},
    {PAD_SNR1_GPIO4, CHIPTOP_BANK, REG_SDIO_MODE,       REG_SDIO_MODE_MASK,     BIT10,      PINMUX_FOR_SDIO_MODE},

    {PAD_SNR1_GPIO5, CHIPTOP_BANK, REG_SR1_PAR_MODE,    REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_GPIO5, CHIPTOP_BANK, REG_I2C1_MODE,       REG_I2C1_MODE_MASK,     BIT6,       PINMUX_FOR_I2C1_MODE},
    {PAD_SNR1_GPIO5, CHIPTOP_BANK, REG_I2C2_MODE,       REG_I2C2_MODE_MASK,     BIT8,       PINMUX_FOR_I2C2_MODE},
    {PAD_SNR1_GPIO5, CHIPTOP_BANK, REG_SDIO_MODE,       REG_SDIO_MODE_MASK,     BIT10,      PINMUX_FOR_SDIO_MODE},

    {PAD_SNR1_GPIO6, CHIPTOP_BANK, REG_SR1_PAR_MODE,    REG_SR1_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR1_PAR_MODE},
    {PAD_SNR1_GPIO6, CHIPTOP_BANK, REG_I2C1_MODE,       REG_I2C1_MODE_MASK,     BIT6,       PINMUX_FOR_I2C1_MODE},
    {PAD_SNR1_GPIO6, CHIPTOP_BANK, REG_I2C2_MODE,       REG_I2C2_MODE_MASK,     BIT8,       PINMUX_FOR_I2C2_MODE},
    {PAD_SNR1_GPIO6, CHIPTOP_BANK, REG_PWM3_MODE,       REG_PWM3_MODE_MASK,     BIT14,      PINMUX_FOR_PWM3_MODE},

    {PAD_NAND_ALE, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_ALE, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_NAND_CLE, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_CLE, CHIPTOP_BANK, REG_SDIO_MODE,         REG_SDIO_MODE_MASK,     BIT9,       PINMUX_FOR_SDIO_MODE},
    {PAD_NAND_CLE, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_CLE, CHIPTOP_BANK, REG_SD_MODE,           REG_SD_MODE_MASK,       BIT2,       PINMUX_FOR_SD_MODE},
    {PAD_NAND_CLE, CHIPTOP_BANK, REG_DMIC_MODE,         REG_DMIC_MODE_MASK,     BIT10,      PINMUX_FOR_DMIC_MODE},
    {PAD_NAND_CLE, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_NAND_CEZ, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_CEZ, CHIPTOP_BANK, REG_SDIO_MODE,         REG_SDIO_MODE_MASK,     BIT9,       PINMUX_FOR_SDIO_MODE},
    {PAD_NAND_CEZ, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_CEZ, CHIPTOP_BANK, REG_SD_MODE,           REG_SD_MODE_MASK,       BIT2,       PINMUX_FOR_SD_MODE},
    {PAD_NAND_CEZ, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_NAND_WEZ, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_WEZ, CHIPTOP_BANK, REG_SDIO_MODE,         REG_SDIO_MODE_MASK,     BIT9,       PINMUX_FOR_SDIO_MODE},
    {PAD_NAND_WEZ, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_WEZ, CHIPTOP_BANK, REG_SD_MODE,           REG_SD_MODE_MASK,       BIT2,       PINMUX_FOR_SD_MODE},
    {PAD_NAND_WEZ, CHIPTOP_BANK, REG_DMIC_MODE,         REG_DMIC_MODE_MASK,     BIT10,      PINMUX_FOR_DMIC_MODE},
    {PAD_NAND_WEZ, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_NAND_WPZ, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_WPZ, CHIPTOP_BANK, REG_SDIO_MODE,         REG_SDIO_MODE_MASK,     BIT9,       PINMUX_FOR_SDIO_MODE},
    {PAD_NAND_WPZ, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_WPZ, CHIPTOP_BANK, REG_SD_MODE,           REG_SD_MODE_MASK,       BIT2,       PINMUX_FOR_SD_MODE},
    {PAD_NAND_WPZ, CHIPTOP_BANK, REG_DMIC_MODE,         REG_DMIC_MODE_MASK,     BIT10,      PINMUX_FOR_DMIC_MODE},
    {PAD_NAND_WPZ, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_NAND_REZ, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_REZ, CHIPTOP_BANK, REG_SDIO_MODE,         REG_SDIO_MODE_MASK,     BIT9,       PINMUX_FOR_SDIO_MODE},
    {PAD_NAND_REZ, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_REZ, CHIPTOP_BANK, REG_SD_MODE,           REG_SD_MODE_MASK,       BIT2,       PINMUX_FOR_SD_MODE},
    {PAD_NAND_REZ, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_NAND_RBZ, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_RBZ, CHIPTOP_BANK, REG_SDIO_MODE,         REG_SDIO_MODE_MASK,     BIT9,       PINMUX_FOR_SDIO_MODE},
    {PAD_NAND_RBZ, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_RBZ, CHIPTOP_BANK, REG_SD_MODE,           REG_SD_MODE_MASK,       BIT2,       PINMUX_FOR_SD_MODE},
    {PAD_NAND_RBZ, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA0, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA0, CHIPTOP_BANK, REG_EMMC_RSTN_EN,      REG_EMMC_RSTN_EN_MASK,  BIT1,       PINMUX_FOR_EMMC_RSTN_EN},
    {PAD_NAND_DA0, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA1, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA1, CHIPTOP_BANK, REG_DMIC_MODE,         REG_DMIC_MODE_MASK,     BIT10|BIT8, PINMUX_FOR_DMIC_MODE},
    {PAD_NAND_DA1, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA2, CHIPTOP_BANK, REG_I2C3_MODE,         REG_I2C3_MODE_MASK,     BIT11,      PINMUX_FOR_I2C3_MODE},
    {PAD_NAND_DA2, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA2, CHIPTOP_BANK, REG_PWM2_MODE,         REG_PWM2_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_PWM2_MODE},
    {PAD_NAND_DA2, CHIPTOP_BANK, REG_DMIC_MODE,         REG_DMIC_MODE_MASK,     BIT10|BIT8, PINMUX_FOR_DMIC_MODE},
    {PAD_NAND_DA2, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA3, CHIPTOP_BANK, REG_I2C3_MODE,         REG_I2C3_MODE_MASK,     BIT11,      PINMUX_FOR_I2C3_MODE},
    {PAD_NAND_DA3, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA3, CHIPTOP_BANK, REG_PWM3_MODE,         REG_PWM3_MODE_MASK,     BIT13|BIT12,PINMUX_FOR_PWM3_MODE},
    {PAD_NAND_DA3, CHIPTOP_BANK, REG_DMIC_MODE,         REG_DMIC_MODE_MASK,     BIT10|BIT8, PINMUX_FOR_DMIC_MODE},
    {PAD_NAND_DA3, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA4, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA4, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_DA4, CHIPTOP_BANK, REG_PWM4_MODE,         REG_PWM4_MODE_MASK,     BIT0,       PINMUX_FOR_PWM4_MODE},
    {PAD_NAND_DA4, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA5, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA5, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_DA5, CHIPTOP_BANK, REG_PWM5_MODE,         REG_PWM5_MODE_MASK,     BIT3,       PINMUX_FOR_PWM5_MODE},
    {PAD_NAND_DA5, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA6, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA6, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_DA6, CHIPTOP_BANK, REG_PWM6_MODE,         REG_PWM6_MODE_MASK,     BIT8,       PINMUX_FOR_PWM6_MODE},
    {PAD_NAND_DA6, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_NAND_DA7, CHIPTOP_BANK, REG_NAND_MODE,         REG_NAND_MODE_MASK,     BIT0,       PINMUX_FOR_NAND_MODE},
    {PAD_NAND_DA7, CHIPTOP_BANK, REG_EMMC_MODE,         REG_EMMC_MODE_MASK,     BIT0,       PINMUX_FOR_EMMC_MODE},
    {PAD_NAND_DA7, CHIPTOP_BANK, REG_PWM7_MODE,         REG_PWM7_MODE_MASK,     BIT12,      PINMUX_FOR_PWM7_MODE},
    {PAD_NAND_DA7, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D0, CHIPTOP_BANK, REG_I2C3_MODE,           REG_I2C3_MODE_MASK,     BIT11,      PINMUX_FOR_I2C3_MODE},
    {PAD_LCD_D0, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT6|BIT5,  PINMUX_FOR_SPI1_MODE},
    {PAD_LCD_D0, CHIPTOP_BANK, REG_SDIO_MODE,           REG_SDIO_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_SDIO_MODE},
    {PAD_LCD_D0, CHIPTOP_BANK, REG_PWM2_MODE,           REG_PWM2_MODE_MASK,     BIT10|BIT9, PINMUX_FOR_PWM2_MODE},
    {PAD_LCD_D0, CHIPTOP_BANK, REG_I2S_MODE,            REG_I2S_MODE_MASK,      BIT13,      PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D0, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D1, CHIPTOP_BANK, REG_I2C3_MODE,           REG_I2C3_MODE_MASK,     BIT11,      PINMUX_FOR_I2C3_MODE},
    {PAD_LCD_D1, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT6|BIT5,  PINMUX_FOR_SPI1_MODE},
    {PAD_LCD_D1, CHIPTOP_BANK, REG_SDIO_MODE,           REG_SDIO_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_SDIO_MODE},
    {PAD_LCD_D1, CHIPTOP_BANK, REG_PWM3_MODE,           REG_PWM3_MODE_MASK,     BIT14|BIT12,PINMUX_FOR_PWM3_MODE},
    {PAD_LCD_D1, CHIPTOP_BANK, REG_I2S_MODE,            REG_I2S_MODE_MASK,      BIT13,      PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D1, CHIPTOP_BANK, REG_DMIC_MODE,           REG_DMIC_MODE_MASK,     BIT9,       PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D1, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D2, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT6|BIT5,  PINMUX_FOR_SPI1_MODE},
    {PAD_LCD_D2, CHIPTOP_BANK, REG_SDIO_MODE,           REG_SDIO_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_SDIO_MODE},
    {PAD_LCD_D2, CHIPTOP_BANK, REG_I2S_MODE,            REG_I2S_MODE_MASK,      BIT13,      PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D2, CHIPTOP_BANK, REG_DMIC_MODE,           REG_DMIC_MODE_MASK,     BIT9,       PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D2, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D3, CHIPTOP_BANK, REG_SPI1_MODE,           REG_SPI1_MODE_MASK,     BIT6|BIT5,  PINMUX_FOR_SPI1_MODE},
    {PAD_LCD_D3, CHIPTOP_BANK, REG_SDIO_MODE,           REG_SDIO_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_SDIO_MODE},
    {PAD_LCD_D3, CHIPTOP_BANK, REG_I2S_MODE,            REG_I2S_MODE_MASK,      BIT13,      PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D3, CHIPTOP_BANK, REG_DMIC_MODE,           REG_DMIC_MODE_MASK,     BIT9,       PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D3, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D4, CHIPTOP_BANK, REG_I2C2_MODE,           REG_I2C2_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_I2C2_MODE},
    {PAD_LCD_D4, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D5, CHIPTOP_BANK, REG_I2C2_MODE,           REG_I2C2_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_I2C2_MODE},
    {PAD_LCD_D5, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D6, CHIPTOP_BANK, REG_I2C3_MODE,           REG_I2C3_MODE_MASK,     BIT12,      PINMUX_FOR_I2C3_MODE},
    {PAD_LCD_D6, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D7, CHIPTOP_BANK, REG_I2C3_MODE,           REG_I2C3_MODE_MASK,     BIT12,      PINMUX_FOR_I2C3_MODE},
    {PAD_LCD_D7, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D8, CHIPTOP_BANK, REG_I2S_MCK_MODE,        REG_I2S_MCK_MODE_MASK,  BIT8,       PINMUX_FOR_I2S_MCK_MODE},
    {PAD_LCD_D8, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D9, CHIPTOP_BANK, REG_I2S_MODE,            REG_I2S_MODE_MASK,      BIT13|BIT12,PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D9, CHIPTOP_BANK, REG_I2S_RX_MODE,         REG_I2S_RX_MODE_MASK,   BIT0,       PINMUX_FOR_I2S_RX_MODE},
    {PAD_LCD_D9, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D10, CHIPTOP_BANK, REG_I2S_MODE,           REG_I2S_MODE_MASK,      BIT13|BIT12,PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D10, CHIPTOP_BANK, REG_I2S_RX_MODE,        REG_I2S_RX_MODE_MASK,   BIT0,       PINMUX_FOR_I2S_RX_MODE},
    {PAD_LCD_D10, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D11, CHIPTOP_BANK, REG_I2S_MODE,           REG_I2S_MODE_MASK,      BIT13|BIT12,PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D11, CHIPTOP_BANK, REG_I2S_RX_MODE,        REG_I2S_RX_MODE_MASK,   BIT0,       PINMUX_FOR_I2S_RX_MODE},
    {PAD_LCD_D11, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D12, CHIPTOP_BANK, REG_I2S_MODE,           REG_I2S_MODE_MASK,      BIT13|BIT12,PINMUX_FOR_I2S_MODE},
    {PAD_LCD_D12, CHIPTOP_BANK, REG_I2S_TX_MODE,        REG_I2S_TX_MODE_MASK,   BIT4,       PINMUX_FOR_I2S_TX_MODE},
    {PAD_LCD_D12, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D13, CHIPTOP_BANK, REG_I2S_TX_MODE,        REG_I2S_TX_MODE_MASK,   BIT4,       PINMUX_FOR_I2S_TX_MODE},
    {PAD_LCD_D13, CHIPTOP_BANK, REG_DMIC_MODE,          REG_DMIC_MODE_MASK,     BIT10|BIT9, PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D13, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D14, CHIPTOP_BANK, REG_I2S_TX_MODE,        REG_I2S_TX_MODE_MASK,   BIT4,       PINMUX_FOR_I2S_TX_MODE},
    {PAD_LCD_D14, CHIPTOP_BANK, REG_DMIC_MODE,          REG_DMIC_MODE_MASK,     BIT10|BIT9, PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D14, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D15, CHIPTOP_BANK, REG_DMIC_MODE,          REG_DMIC_MODE_MASK,     BIT10|BIT9|BIT8, PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D15, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D16, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D17, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D18, CHIPTOP_BANK, REG_DMIC_MODE,          REG_DMIC_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D18, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D19, CHIPTOP_BANK, REG_DMIC_MODE,          REG_DMIC_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_DMIC_MODE},
    {PAD_LCD_D19, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D20, CHIPTOP_BANK, REG_FUART_MODE,         REG_FUART_MODE_MASK,    BIT2,       PINMUX_FOR_FUART_MODE},
    {PAD_LCD_D20, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D21, CHIPTOP_BANK, REG_FUART_MODE,         REG_FUART_MODE_MASK,    BIT2,       PINMUX_FOR_FUART_MODE},
    {PAD_LCD_D21, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D22, CHIPTOP_BANK, REG_FUART_MODE,         REG_FUART_MODE_MASK,    BIT2,       PINMUX_FOR_FUART_MODE},
    {PAD_LCD_D22, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_D23, CHIPTOP_BANK, REG_FUART_MODE,         REG_FUART_MODE_MASK,    BIT2,       PINMUX_FOR_FUART_MODE},
    {PAD_LCD_D23, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT7,       PINMUX_FOR_TTL_MODE},

    {PAD_LCD_VSYNC, CHIPTOP_BANK, REG_SR0_PAR_MODE,     REG_SR0_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR0_PAR_MODE},
    {PAD_LCD_VSYNC, CHIPTOP_BANK, REG_PWM0_MODE,        REG_PWM0_MODE_MASK,     BIT2,       PINMUX_FOR_PWM0_MODE},
    {PAD_LCD_VSYNC, CHIPTOP_BANK, REG_TTL_MODE,         REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_HSYNC, CHIPTOP_BANK, REG_SR0_PAR_MODE,     REG_SR0_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR0_PAR_MODE},
    {PAD_LCD_HSYNC, CHIPTOP_BANK, REG_SDIO_MODE,        REG_SDIO_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_SDIO_MODE},
    {PAD_LCD_HSYNC, CHIPTOP_BANK, REG_PWM1_MODE,        REG_PWM1_MODE_MASK,     BIT5,       PINMUX_FOR_PWM1_MODE},
    {PAD_LCD_HSYNC, CHIPTOP_BANK, REG_TTL_MODE,         REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_PCLK, CHIPTOP_BANK, REG_SR0_PAR_MODE,      REG_SR0_PAR_MODE_MASK,  BIT10,      PINMUX_FOR_SR0_PAR_MODE},
    {PAD_LCD_PCLK, CHIPTOP_BANK, REG_SDIO_MODE,         REG_SDIO_MODE_MASK,     BIT9|BIT8,  PINMUX_FOR_SDIO_MODE},
    {PAD_LCD_PCLK, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_LCD_DE, CHIPTOP_BANK, REG_TTL_MODE,            REG_TTL_MODE_MASK,      BIT7|BIT6,  PINMUX_FOR_TTL_MODE},

    {PAD_UART0_RX, CHIPTOP_BANK, REG_I2C2_MODE,         REG_I2C2_MODE_MASK,     BIT9,       PINMUX_FOR_I2C2_MODE},
    {PAD_UART0_RX, CHIPTOP_BANK, REG_I2C3_MODE,         REG_I2C3_MODE_MASK,     BIT12|BIT10,PINMUX_FOR_I2C3_MODE},
    {PAD_UART0_RX, CHIPTOP_BANK, REG_UART0_MODE,        REG_UART0_MODE_MASK,    BIT4,       PINMUX_FOR_UART0_MODE},
    {PAD_UART0_RX, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_UART0_TX, CHIPTOP_BANK, REG_I2C2_MODE,         REG_I2C2_MODE_MASK,     BIT9,       PINMUX_FOR_I2C2_MODE},
    {PAD_UART0_TX, CHIPTOP_BANK, REG_I2C3_MODE,         REG_I2C3_MODE_MASK,     BIT12|BIT10,PINMUX_FOR_I2C3_MODE},
    {PAD_UART0_TX, CHIPTOP_BANK, REG_UART0_MODE,        REG_UART0_MODE_MASK,    BIT4,       PINMUX_FOR_UART0_MODE},
    {PAD_UART0_TX, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_UART1_RX, CHIPTOP_BANK, REG_UART1_MODE,        REG_UART1_MODE_MASK,    BIT8,       PINMUX_FOR_UART1_MODE},
    {PAD_UART1_RX, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_UART1_TX, CHIPTOP_BANK, REG_UART1_MODE,        REG_UART1_MODE_MASK,    BIT8,       PINMUX_FOR_UART1_MODE},
    {PAD_UART1_TX, CHIPTOP_BANK, REG_TTL_MODE,          REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI0_CZ, CHIPTOP_BANK, REG_EJ_MODE,            REG_EJ_MODE_MASK,       BIT1,       PINMUX_FOR_EJ_MODE},
    {PAD_SPI0_CZ, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT1|BIT0,  PINMUX_FOR_TESTIN_MODE},
    {PAD_SPI0_CZ, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT4,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SPI0_CZ, CHIPTOP_BANK, REG_SPI0_MODE,          REG_SPI0_MODE_MASK,     BIT0,       PINMUX_FOR_SPI0_MODE},
    {PAD_SPI0_CZ, CHIPTOP_BANK, REG_PWM4_MODE,          REG_PWM4_MODE_MASK,     BIT1,       PINMUX_FOR_PWM4_MODE},
    {PAD_SPI0_CZ, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI0_CK, CHIPTOP_BANK, REG_EJ_MODE,            REG_EJ_MODE_MASK,       BIT1,       PINMUX_FOR_EJ_MODE},
    {PAD_SPI0_CK, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT1|BIT0,  PINMUX_FOR_TESTIN_MODE},
    {PAD_SPI0_CK, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT4,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SPI0_CK, CHIPTOP_BANK, REG_SPI0_MODE,          REG_SPI0_MODE_MASK,     BIT0,       PINMUX_FOR_SPI0_MODE},
    {PAD_SPI0_CK, CHIPTOP_BANK, REG_PWM5_MODE,          REG_PWM5_MODE_MASK,     BIT4,       PINMUX_FOR_PWM5_MODE},
    {PAD_SPI0_CK, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI0_DI, CHIPTOP_BANK, REG_EJ_MODE,            REG_EJ_MODE_MASK,       BIT1,       PINMUX_FOR_EJ_MODE},
    {PAD_SPI0_DI, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT1|BIT0,  PINMUX_FOR_TESTIN_MODE},
    {PAD_SPI0_DI, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT4,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SPI0_DI, CHIPTOP_BANK, REG_SPI0_MODE,          REG_SPI0_MODE_MASK,     BIT0,       PINMUX_FOR_SPI0_MODE},
    {PAD_SPI0_DI, CHIPTOP_BANK, REG_PWM6_MODE,          REG_PWM6_MODE_MASK,     BIT9,       PINMUX_FOR_PWM6_MODE},
    {PAD_SPI0_DI, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI0_DO, CHIPTOP_BANK, REG_EJ_MODE,            REG_EJ_MODE_MASK,       BIT1,       PINMUX_FOR_EJ_MODE},
    {PAD_SPI0_DO, CHIPTOP_BANK, REG_TESTIN_MODE,        REG_TESTIN_MODE_MASK,   BIT1|BIT0,  PINMUX_FOR_TESTIN_MODE},
    {PAD_SPI0_DO, CHIPTOP_BANK, REG_TESTOUT_MODE,       REG_TESTOUT_MODE_MASK,  BIT4,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SPI0_DO, CHIPTOP_BANK, REG_SPI0_MODE,          REG_SPI0_MODE_MASK,     BIT0,       PINMUX_FOR_SPI0_MODE},
    {PAD_SPI0_DO, CHIPTOP_BANK, REG_PWM7_MODE,          REG_PWM7_MODE_MASK,     BIT13,       PINMUX_FOR_PWM7_MODE},
    {PAD_SPI0_DO, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI1_CZ, CHIPTOP_BANK, REG_SPI1_MODE,          REG_SPI1_MODE_MASK,     BIT4,       PINMUX_FOR_SPI1_MODE},
    {PAD_SPI1_CZ, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI1_CK, CHIPTOP_BANK, REG_SPI1_MODE,          REG_SPI1_MODE_MASK,     BIT4,       PINMUX_FOR_SPI1_MODE},
    {PAD_SPI1_CK, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI1_DI, CHIPTOP_BANK, REG_SPI1_MODE,          REG_SPI1_MODE_MASK,     BIT4,       PINMUX_FOR_SPI1_MODE},
    {PAD_SPI1_DI, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_SPI1_DO, CHIPTOP_BANK, REG_SPI1_MODE,          REG_SPI1_MODE_MASK,     BIT4,       PINMUX_FOR_SPI1_MODE},
    {PAD_SPI1_DO, CHIPTOP_BANK, REG_TTL_MODE,           REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_PWM0, CHIPTOP_BANK, REG_I2C0_MODE,             REG_I2C0_MODE_MASK,     BIT1,       PINMUX_FOR_I2C0_MODE},
    {PAD_PWM0, CHIPTOP_BANK, REG_I2C1_MODE,             REG_I2C1_MODE_MASK,     BIT5,       PINMUX_FOR_I2C1_MODE},
    {PAD_PWM0, CHIPTOP_BANK, REG_PWM0_MODE,             REG_PWM0_MODE_MASK,     BIT0,       PINMUX_FOR_PWM0_MODE},
    {PAD_PWM0, CHIPTOP_BANK, REG_PWM2_MODE,             REG_PWM2_MODE_MASK,     BIT10|BIT9|BIT8, PINMUX_FOR_PWM2_MODE},
    {PAD_PWM0, CHIPTOP_BANK, REG_TTL_MODE,              REG_TTL_MODE_MASK,      BIT6,       PINMUX_FOR_TTL_MODE},

    {PAD_PWM1, CHIPTOP_BANK, REG_I2C0_MODE,             REG_I2C0_MODE_MASK,     BIT1,       PINMUX_FOR_I2C0_MODE},
    {PAD_PWM1, CHIPTOP_BANK, REG_I2C1_MODE,             REG_I2C1_MODE_MASK,     BIT5,       PINMUX_FOR_I2C1_MODE},
    {PAD_PWM1, CHIPTOP_BANK, REG_PWM1_MODE,             REG_PWM1_MODE_MASK,     BIT3,       PINMUX_FOR_PWM1_MODE},
    {PAD_PWM1, CHIPTOP_BANK, REG_PWM3_MODE,             REG_PWM3_MODE_MASK,     BIT14|BIT13,PINMUX_FOR_PWM3_MODE},

    {PAD_SD_CLK, CHIPTOP_BANK, REG_TESTIN_MODE,         REG_TESTIN_MODE_MASK,   BIT1,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SD_CLK, CHIPTOP_BANK, REG_TESTOUT_MODE,        REG_TESTOUT_MODE_MASK,  BIT5,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SD_CLK, CHIPTOP_BANK, REG_SDIO_MODE,           REG_SDIO_MODE_MASK,     BIT8,       PINMUX_FOR_SDIO_MODE},
    {PAD_SD_CLK, CHIPTOP_BANK, REG_SD_MODE,             REG_SD_MODE_MASK,       BIT3,       PINMUX_FOR_SD_MODE},

    {PAD_SD_CMD, CHIPTOP_BANK, REG_TESTIN_MODE,         REG_TESTIN_MODE_MASK,   BIT1,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SD_CMD, CHIPTOP_BANK, REG_TESTOUT_MODE,        REG_TESTOUT_MODE_MASK,  BIT5,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SD_CMD, CHIPTOP_BANK, REG_SDIO_MODE,           REG_SDIO_MODE_MASK,     BIT8,       PINMUX_FOR_SDIO_MODE},
    {PAD_SD_CMD, CHIPTOP_BANK, REG_SD_MODE,             REG_SD_MODE_MASK,       BIT3,       PINMUX_FOR_SD_MODE},

    {PAD_SD_D0, CHIPTOP_BANK, REG_TESTIN_MODE,          REG_TESTIN_MODE_MASK,   BIT1,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SD_D0, CHIPTOP_BANK, REG_TESTOUT_MODE,         REG_TESTOUT_MODE_MASK,  BIT5,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SD_D0, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT5|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SD_D0, CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,     BIT8,       PINMUX_FOR_SDIO_MODE},
    {PAD_SD_D0, CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,       BIT3,       PINMUX_FOR_SD_MODE},

    {PAD_SD_D1, CHIPTOP_BANK, REG_TESTIN_MODE,          REG_TESTIN_MODE_MASK,   BIT1,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SD_D1, CHIPTOP_BANK, REG_TESTOUT_MODE,         REG_TESTOUT_MODE_MASK,  BIT5,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SD_D1, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT5|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SD_D1, CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,     BIT8,       PINMUX_FOR_SDIO_MODE},
    {PAD_SD_D1, CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,       BIT3,       PINMUX_FOR_SD_MODE},

    {PAD_SD_D2, CHIPTOP_BANK, REG_TESTIN_MODE,          REG_TESTIN_MODE_MASK,   BIT1,       PINMUX_FOR_TESTIN_MODE},
    {PAD_SD_D2, CHIPTOP_BANK, REG_TESTOUT_MODE,         REG_TESTOUT_MODE_MASK,  BIT5,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SD_D2, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT5|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SD_D2, CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,     BIT8,       PINMUX_FOR_SDIO_MODE},
    {PAD_SD_D2, CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,       BIT3,       PINMUX_FOR_SD_MODE},

    {PAD_SD_D3, CHIPTOP_BANK, REG_TESTIN_MODE,          REG_TESTIN_MODE_MASK,   BIT1|BIT0,  PINMUX_FOR_TESTIN_MODE},
    {PAD_SD_D3, CHIPTOP_BANK, REG_TESTOUT_MODE,         REG_TESTOUT_MODE_MASK,  BIT5,       PINMUX_FOR_TESTOUT_MODE},
    {PAD_SD_D3, CHIPTOP_BANK, REG_SPI1_MODE,            REG_SPI1_MODE_MASK,     BIT5|BIT4,  PINMUX_FOR_SPI1_MODE},
    {PAD_SD_D3, CHIPTOP_BANK, REG_SDIO_MODE,            REG_SDIO_MODE_MASK,     BIT8,       PINMUX_FOR_SDIO_MODE},
    {PAD_SD_D3, CHIPTOP_BANK, REG_SD_MODE,              REG_SD_MODE_MASK,       BIT3,       PINMUX_FOR_SD_MODE},
};

static const ST_PadModeInfo m_stPadModeInfoTbl[] =
{
    {"GPIO",    0,                                              0},
    // Non PM
    {"FUART",   _RIUA_16BIT(CHIPTOP_BANK,REG_FUART_MODE),       REG_FUART_MODE_MASK},
    {"UART0",   _RIUA_16BIT(CHIPTOP_BANK,REG_UART0_MODE),       REG_UART0_MODE_MASK},
    {"UART1",   _RIUA_16BIT(CHIPTOP_BANK,REG_UART1_MODE),       REG_UART1_MODE_MASK},
    {"PWM0",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM0_MODE),        REG_PWM0_MODE_MASK},
    {"PWM1",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM1_MODE),        REG_PWM1_MODE_MASK},
    {"PWM2",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM2_MODE),        REG_PWM2_MODE_MASK},
    {"PWM3",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM3_MODE),        REG_PWM3_MODE_MASK},
    {"PWM4",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM4_MODE),        REG_PWM4_MODE_MASK},
    {"PWM5",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM5_MODE),        REG_PWM5_MODE_MASK},
    {"PWM6",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM6_MODE),        REG_PWM6_MODE_MASK},
    {"PWM7",    _RIUA_16BIT(CHIPTOP_BANK,REG_PWM7_MODE),        REG_PWM7_MODE_MASK},
    {"SR",      _RIUA_16BIT(CHIPTOP_BANK,REG_SR_MODE),          REG_SR_MODE_MASK},
    {"NAND",    _RIUA_16BIT(CHIPTOP_BANK,REG_NAND_MODE),        REG_NAND_MODE_MASK},
    {"SD",      _RIUA_16BIT(CHIPTOP_BANK,REG_SD_MODE),          REG_SD_MODE_MASK},
    {"SDIO",    _RIUA_16BIT(CHIPTOP_BANK,REG_SDIO_MODE),        REG_SDIO_MODE_MASK},
    {"I2C0",    _RIUA_16BIT(CHIPTOP_BANK,REG_I2C0_MODE),        REG_I2C0_MODE_MASK},
    {"I2C1",    _RIUA_16BIT(CHIPTOP_BANK,REG_I2C1_MODE),        REG_I2C1_MODE_MASK},
    {"SPI0",    _RIUA_16BIT(CHIPTOP_BANK,REG_SPI0_MODE),        REG_SPI0_MODE_MASK},
    {"SPI1",    _RIUA_16BIT(CHIPTOP_BANK,REG_SPI1_MODE),        REG_SPI1_MODE_MASK},
    {"EJ",      _RIUA_16BIT(CHIPTOP_BANK,REG_EJ_MODE),          REG_EJ_MODE_MASK},
    {"ETH",     _RIUA_16BIT(CHIPTOP_BANK,REG_ETH_MODE),         REG_ETH_MODE_MASK},
    {"TTL",     _RIUA_16BIT(CHIPTOP_BANK,REG_TTL_MODE),         REG_TTL_MODE_MASK},
    {"DMIC",    _RIUA_16BIT(CHIPTOP_BANK,REG_DMIC_MODE),        REG_DMIC_MODE_MASK},
    {"I2S",     _RIUA_16BIT(CHIPTOP_BANK,REG_I2S_MODE),         REG_I2S_MODE_MASK},
    {"I2S_RX",  _RIUA_16BIT(CHIPTOP_BANK,REG_I2S_RX_MODE),      REG_I2S_RX_MODE_MASK},
    {"I2S_TX",  _RIUA_16BIT(CHIPTOP_BANK,REG_I2S_TX_MODE),      REG_I2S_TX_MODE_MASK},
    {"I2S_MCK", _RIUA_16BIT(CHIPTOP_BANK,REG_I2S_MCK_MODE),     REG_I2S_MCK_MODE_MASK},
    {"TESTIN",  _RIUA_16BIT(CHIPTOP_BANK,REG_TESTIN_MODE),      REG_TESTIN_MODE_MASK},
    {"TESTOUT", _RIUA_16BIT(CHIPTOP_BANK,REG_TESTOUT_MODE),     REG_TESTOUT_MODE_MASK},
    {"EMMC",    _RIUA_16BIT(CHIPTOP_BANK,REG_EMMC_MODE),        REG_EMMC_MODE_MASK},
    {"EMMC_RSTN",_RIUA_16BIT(CHIPTOP_BANK,REG_EMMC_RSTN_EN),    REG_EMMC_RSTN_EN_MASK},
    {"SR0_BT656",_RIUA_16BIT(CHIPTOP_BANK,REG_SR0_BT656_MODE),  REG_SR0_BT656_MODE_MASK},
    {"SR0_MIPI",_RIUA_16BIT(CHIPTOP_BANK,REG_SR0_MIPI_MODE),    REG_SR0_MIPI_MODE_MASK},
    {"SR0_PAR", _RIUA_16BIT(CHIPTOP_BANK,REG_SR0_PAR_MODE),     REG_SR0_PAR_MODE_MASK},
    {"I2C2",    _RIUA_16BIT(CHIPTOP_BANK,REG_I2C2_MODE),        REG_I2C2_MODE_MASK},
    {"I2C3",    _RIUA_16BIT(CHIPTOP_BANK,REG_I2C3_MODE),        REG_I2C3_MODE_MASK},
    {"SR1_BT656",_RIUA_16BIT(CHIPTOP_BANK,REG_SR1_BT656_MODE),  REG_SR1_BT656_MODE_MASK},
    {"SR1_MIPI",_RIUA_16BIT(CHIPTOP_BANK,REG_SR1_MIPI_MODE),    REG_SR1_MIPI_MODE_MASK},
    {"SR1_PAR", _RIUA_16BIT(CHIPTOP_BANK,REG_SR1_PAR_MODE),     REG_SR1_PAR_MODE_MASK},
    // PM Sleep
    {"PM_IR_GPIO",_RIUA_16BIT(PMSLEEP_BANK,REG_PM_IR_IS_GPIO),  REG_PM_IR_IS_GPIO_MASK},
    {"PM_PWM0", _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM0_MODE),     REG_PM_PWM0_MODE_MASK},
    {"PM_PWM1", _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM1_MODE),     REG_PM_PWM1_MODE_MASK},
    {"PM_LED",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_LED_MODE),      REG_PM_LED_MODE_MASK},
    {"PM_PWM2", _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM2_MODE),     REG_PM_PWM2_MODE_MASK},
    {"PM_PWM3", _RIUA_16BIT(PMSLEEP_BANK,REG_PM_PWM3_MODE),     REG_PM_PWM3_MODE_MASK},
    {"PM_VID",  _RIUA_16BIT(PMSLEEP_BANK,REG_PM_VID_MODE),      REG_PM_VID_MODE_MASK},
    {"PM_SD_CDZ",_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SD_CDZ_MODE),  REG_PM_SD_CDZ_MODE_MASK},
    {"PM_SPI_GPIO",_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO),REG_PM_SPI_IS_GPIO_MASK},
    {"PM_UART_GPIO",_RIUA_16BIT(PMSLEEP_BANK,REG_PM_UART_IS_GPIO), REG_PM_UART_IS_GPIO_MASK},
};

//==============================================================================
//
//                              FUNCTIONS
//
//==============================================================================

//------------------------------------------------------------------------------
//  Function    : _HalCheckPin
//  Description :
//------------------------------------------------------------------------------
static S32 _HalCheckPin(U32 padID)
{
    if (GPIO_NR <= padID) {
        return FALSE;
    }
    return TRUE;
}

static void _HalSARGPIOWriteRegBit(u32 u32RegOffset, bool bEnable, U8 u8BitMsk)
{
    if (bEnable)
        _GPIO_R_BYTE(_RIUA_8BIT(SAR_BANK, u32RegOffset)) |= u8BitMsk;
    else
        _GPIO_R_BYTE(_RIUA_8BIT(SAR_BANK, u32RegOffset)) &= (~u8BitMsk);
}

static void _HalPadDisablePadMux(U32 u32PadModeID)
{
    if (_GPIO_R_WORD_MASK(m_stPadModeInfoTbl[u32PadModeID].u32ModeRIU, m_stPadModeInfoTbl[u32PadModeID].u32ModeMask)) {
        _GPIO_W_WORD_MASK(m_stPadModeInfoTbl[u32PadModeID].u32ModeRIU, 0, m_stPadModeInfoTbl[u32PadModeID].u32ModeMask);
    }
}

static S32 HalPadSetMode_NonPM(U32 u32PadID, U32 u32Mode)
{
    U32 u32RegAddr = 0;
    U16 u16RegVal  = 0;
    U8  u8ModeIsFind = 0;
    U16 i = 0;

    for (i = 0; i < sizeof(m_stNonPMPadMuxTbl)/sizeof(struct stPadMux); i++)
    {
        if (u32PadID == m_stNonPMPadMuxTbl[i].padID)
        {
            u32RegAddr = _RIUA_16BIT(m_stNonPMPadMuxTbl[i].base, m_stNonPMPadMuxTbl[i].offset);

            if (u32Mode == m_stNonPMPadMuxTbl[i].mode)
            {
                u16RegVal = _GPIO_R_WORD_MASK(u32RegAddr, 0xFFFF);
                u16RegVal &= ~(m_stNonPMPadMuxTbl[i].mask);
                u16RegVal |= m_stNonPMPadMuxTbl[i].val; // CHECK Multi-Pad Mode

                _GPIO_W_WORD_MASK(u32RegAddr, u16RegVal, 0xFFFF);

                u8ModeIsFind = 1;
                break;
            }
            else
            {
                u16RegVal = _GPIO_R_WORD_MASK(u32RegAddr, m_stNonPMPadMuxTbl[i].mask);

                if (u16RegVal == m_stNonPMPadMuxTbl[i].val) {
                    _GPIO_W_WORD_MASK(u32RegAddr, 0, m_stNonPMPadMuxTbl[i].mask);
                }
            }
        }
    }

    return (u8ModeIsFind) ? TRUE : FALSE;
}

static S32 HalPadSetMode_PM(U32 u32PadID, U32 u32Mode)
{
    switch(u32PadID)
    {
    case PAD_PM_SD_CDZ:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_PM_SD_CDZ_MODE);
        }
        else { //enable CDZ function, so ISR can work
            _GPIO_W_WORD_MASK(m_stPadModeInfoTbl[PINMUX_FOR_PM_SD_CDZ_MODE].u32ModeRIU,
                              m_stPadModeInfoTbl[PINMUX_FOR_PM_SD_CDZ_MODE].u32ModeMask,
                              m_stPadModeInfoTbl[PINMUX_FOR_PM_SD_CDZ_MODE].u32ModeMask);
        }
        break;
    case PAD_PM_IRIN:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_IR_IS_GPIO), REG_PM_IR_IS_GPIO_MASK, REG_PM_IR_IS_GPIO_MASK);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_IR_IS_GPIO), ~REG_PM_IR_IS_GPIO_MASK, REG_PM_IR_IS_GPIO_MASK);
        }
       break;
    case PAD_PM_GPIO0:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM0_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_PM_VID_MODE);
        }
        break;
    case PAD_PM_GPIO1:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM1_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_UART_IS_GPIO), ~REG_PM_IR_IS_GPIO_MASK, REG_PM_IR_IS_GPIO_MASK);
        }
        break;
    case PAD_PM_GPIO2:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM2_MODE);
        }
        break;
    case PAD_PM_GPIO3:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM3_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_PM_VID_MODE);
        }
        break;
    case PAD_PM_GPIO4:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_LINK_WKINT2GPIO4), ~REG_PM_LINK_WKINT2GPIO4_MASK, REG_PM_LINK_WKINT2GPIO4_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM4_INV), ~REG_PM_GPIO_PM4_INV_MASK, REG_PM_GPIO_PM4_INV_MASK);
            _GPIO_W_WORD(_RIUA_16BIT(PMSLEEP_BANK,REG_GPIO_PM_LOCK), 0xBABE);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_LINK_WKINT2GPIO4), REG_PM_LINK_WKINT2GPIO4_MASK, REG_PM_LINK_WKINT2GPIO4_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_GPIO_PM4_INV), REG_PM_GPIO_PM4_INV_MASK, REG_PM_GPIO_PM4_INV_MASK);
            _GPIO_W_WORD(_RIUA_16BIT(PMSLEEP_BANK,REG_GPIO_PM_LOCK), 0x0);
        }
        break;
    case PAD_PM_GPIO5:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM1_MODE);
        }
        break;
    case PAD_PM_GPIO6:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM0_MODE);
        }
        break;
    case PAD_PM_GPIO7:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_VID_MODE);
        }
        break;
    case PAD_PM_GPIO8:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            if ((Chip_Get_Storage_Type() == MS_STORAGE_SPINAND_ECC) || (Chip_Get_Storage_Type() == MS_STORAGE_SPINAND_NOECC))
            {
                _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT3, BIT3|BIT5|BIT8);
            }
            else
            {
                _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT3|BIT5, BIT2|BIT3|BIT5|BIT8);
            }
            _HalPadDisablePadMux(PINMUX_FOR_PM_UART_IS_GPIO);
        }
        else {
            //_GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT2, BIT2);
            //_GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT3, BIT3);
            //_GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT5, BIT5);
            //_GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT8, BIT8);
        }
        break;
    case PAD_PM_GPIO9:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM2_MODE);
        }
        break;
    case PAD_PM_GPIO10:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_PWM3_MODE);
        }
        break;
    case PAD_PM_GPIO11:
    case PAD_PM_GPIO12:
        break;
    case PAD_PM_GPIO13:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT3, BIT3);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT5, BIT5);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT3, BIT3);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT5, BIT5);
        }
        break;
    case PAD_PM_GPIO14:
    case PAD_PM_GPIO15:
        break;
    case PAD_PM_SPI_CZ:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT2, BIT2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT3, BIT3);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT5, BIT5);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT2, BIT2);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT3, BIT3);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT5, BIT5);
        }
        break;
    case PAD_PM_SPI_CK:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT0, BIT0);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT0, BIT0);
        }
        break;
    case PAD_PM_SPI_DI:
    case PAD_PM_SPI_DO:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT0, BIT0);
        }
        else {
            //CHECK
            //_GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT0|BIT1|BIT2, BIT0|BIT1|BIT2);
        }
        break;
    case PAD_PM_SPI_WPZ:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT4, BIT4);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), ~BIT4, BIT4);
        }
        break;
    case PAD_PM_SPI_HLD:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), BIT6, BIT6|BIT7);
        }
        else {
            //CHECK
            //_GPIO_W_WORD_MASK(_RIUA_16BIT(PMSLEEP_BANK,REG_PM_SPI_IS_GPIO), 0, BIT6|BIT7);
        }
        break;
    case PAD_PM_LED0:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_VID_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_PM_LED_MODE);
        }
        break;
    case PAD_PM_LED1:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_PM_VID_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_PM_LED_MODE);
        }
        break;
    case PAD_SAR_GPIO0:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, 0, REG_SAR_CH0_AISEL);
        }
        else {
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, REG_SAR_CH0_AISEL, REG_SAR_CH0_AISEL);
        }
        break;
    case PAD_SAR_GPIO1:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, 0, REG_SAR_CH1_AISEL);
        }
        else {
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, REG_SAR_CH1_AISEL, REG_SAR_CH1_AISEL);
        }
        break;
    case PAD_SAR_GPIO2:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, 0, REG_SAR_CH2_AISEL);
        }
        else {
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, REG_SAR_CH2_AISEL, REG_SAR_CH2_AISEL);
        }
        break;
    case PAD_SAR_GPIO3:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, 0, REG_SAR_CH3_AISEL);
        }
        else {
            _HalSARGPIOWriteRegBit(REG_SAR_AISEL, REG_SAR_CH3_AISEL, REG_SAR_CH3_AISEL);
        }
        break;
    case PAD_ETH_RN:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY1_BANK,REG_ATOP_RX_INOFF), BIT14, BIT14);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT0, BIT0);
        }
        break;
    case PAD_ETH_RP:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY1_BANK,REG_ATOP_RX_INOFF), BIT14, BIT14);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT1, BIT1);
        }
        break;
    case PAD_ETH_TN:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY1_BANK,REG_ATOP_RX_INOFF), BIT15, BIT15);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT2, BIT2);
        }
        break;
    case PAD_ETH_TP:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY1_BANK,REG_ATOP_RX_INOFF), BIT15, BIT15);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(ALBANY2_BANK,REG_ETH_GPIO_EN), BIT3, BIT3);
        }
        break;
    case PAD_USB_DM:
    case PAD_USB_DP:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _HalPadDisablePadMux(PINMUX_FOR_TESTIN_MODE);
            _HalPadDisablePadMux(PINMUX_FOR_TESTOUT_MODE);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_GPIO_EN), REG_UTMI0_GPIO_EN_MASK, REG_UTMI0_GPIO_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_CLK_EXTRA0_EN), ~REG_UTMI0_CLK_EXTRA0_EN_MASK, REG_UTMI0_CLK_EXTRA0_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_REG_PDN), ~REG_UTMI0_REG_PDN_MASK, REG_UTMI0_REG_PDN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_FL_XVR_PDN), ~REG_UTMI0_FL_XVR_PDN_MASK, REG_UTMI0_FL_XVR_PDN_MASK);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_GPIO_EN), ~REG_UTMI0_GPIO_EN_MASK, REG_UTMI0_GPIO_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_CLK_EXTRA0_EN), REG_UTMI0_CLK_EXTRA0_EN_MASK, REG_UTMI0_CLK_EXTRA0_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_REG_PDN), REG_UTMI0_REG_PDN_MASK, REG_UTMI0_REG_PDN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI0_BANK,REG_UTMI0_FL_XVR_PDN), REG_UTMI0_FL_XVR_PDN_MASK, REG_UTMI0_FL_XVR_PDN_MASK);
        }
        break;
    case PAD_DM_P1:
    case PAD_DP_P1:
        if (u32Mode == PINMUX_FOR_GPIO_MODE) {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_GPIO_EN), REG_UTMI1_GPIO_EN_MASK, REG_UTMI1_GPIO_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_CLK_EXTRA0_EN), ~REG_UTMI1_CLK_EXTRA0_EN_MASK, REG_UTMI1_CLK_EXTRA0_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_REG_PDN), ~REG_UTMI1_REG_PDN_MASK, REG_UTMI1_REG_PDN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_FL_XVR_PDN), ~REG_UTMI1_FL_XVR_PDN_MASK, REG_UTMI1_FL_XVR_PDN_MASK);
        }
        else {
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_GPIO_EN), ~REG_UTMI1_GPIO_EN_MASK, REG_UTMI1_GPIO_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_CLK_EXTRA0_EN), REG_UTMI1_CLK_EXTRA0_EN_MASK, REG_UTMI1_CLK_EXTRA0_EN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_REG_PDN), REG_UTMI1_REG_PDN_MASK, REG_UTMI1_REG_PDN_MASK);
            _GPIO_W_WORD_MASK(_RIUA_16BIT(UTMI1_BANK,REG_UTMI1_FL_XVR_PDN), REG_UTMI1_FL_XVR_PDN_MASK, REG_UTMI1_FL_XVR_PDN_MASK);
        }
        break;
    default:
        break;
    }

    return TRUE;
}

//------------------------------------------------------------------------------
//  Function    : HalPadSetVal
//  Description :
//------------------------------------------------------------------------------
S32 HalPadSetVal(U32 u32PadID, U32 u32Mode)
{
    if (FALSE == _HalCheckPin(u32PadID)) {
        return FALSE;
    }

    if (u32PadID >= PAD_GPIO0 && u32PadID <= PAD_SD_D3) {
        return HalPadSetMode_NonPM(u32PadID, u32Mode);
    }
    else {
        return HalPadSetMode_PM(u32PadID, u32Mode);
    }
}
